from openai import OpenAI
import os
import logging
import json
from tqdm import tqdm
from concurrent.futures import ThreadPoolExecutor, as_completed

QINGYUN_API_KEY = os.environ.get('', '')
QINGYUN_BASE_URL = ""

ANALYSIS_PROMPT_TEMPLATE = """
            You are an expert in Operations Research (OR). You will be given an optimization problem and (optionally) a step-by-step solution, which may or may not include code.

            Task:
            Review the solution. Analyze each applicable part in order. Be concise — only highlight critical errors or omissions. Skip any section if the input doesn't contain it (e.g., no code → skip Code Analysis).

            Evaluate in this order:

            1. Variable Definitions
            2. Objective Function & Constraints
            3. Code Implementation (if provided)
            4. Final Answer / Output

            Question:
            {Question}

            Solution Steps:
            {Solution}

            Output Format (be brief and precise):

            1. Variable Definition Analysis
            - Intent: [e.g., Define decision variables]
            - Analysis: [Only note missing, redundant, or misdefined variables]
            - Judgement: [Correct/Incorrect]

            2. Objective & Constraint Analysis
            - Intent: [e.g., Formulate model]
            - Objective: [Correct? Brief reason if wrong]
            - Constraints: [Missing/incorrect? List only key issues]
            - Judgement: [Correct/Incorrect]

            3. Code Analysis (Skip if no code)
            - Intent: Implement model in Pyomo/Python
            - Analysis: [Only flag mismatches: missing vars/constraints, wrong indexing, type errors]
            - Judgement: [Correct/Incorrect or Skipped]

            4. Final Answer Analysis
            - Intent: [e.g., Report solution or error]
            - Analysis: [Plausible? Error meaningful? Root cause if wrong]
            - Judgement: [Correct/Incorrect]

            Corrected Step (Only if any part above is Incorrect)
            - [Rewrite only the first incorrect section — e.g., fix constraints or variables — in full, clearly labeled.]
            """

def get_or_analysis(question: str,
                    solution: str,
                    model: str = "gpt-4o",
                    api_key: str = QINGYUN_API_KEY,
                    base_url: str = QINGYUN_BASE_URL,
                    timeout: int = 120) -> str:
    try:
        final_prompt = ANALYSIS_PROMPT_TEMPLATE.format(Question=question, Solution=solution)

        client = OpenAI(
            base_url=base_url,
            api_key=api_key,
            timeout=timeout
        )

        messages = [
            {"role": "user", "content": final_prompt}
        ]

        response = client.chat.completions.create(
            model=model,
            messages=messages
        )

        analysis_result = response.choices[0].message.content
        return analysis_result

    except Exception as e:
        print(f"An error occurred: {e}")
        return None

def process_single_item(d, output_dir):
    question = d['question']
    process = "\n\n".join(d['process'])
    hash_value = hash(question + process)
    filename = os.path.join(output_dir, f'{hash_value}.json')

    if os.path.exists(filename):
        return hash_value 

    result = get_or_analysis(question, process)
    d['critic'] = result

    with open(filename, 'w') as f:
        json.dump(d, f, indent=4, ensure_ascii=False)

    return hash_value

if __name__ == '__main__':
    output_dir = ""
    os.makedirs(output_dir, exist_ok=True) 

    data = json.load(open(""))

    
    items_to_process = data

    max_workers = 16

    with ThreadPoolExecutor(max_workers=max_workers) as executor:
        future_to_item = {
            executor.submit(process_single_item, d, output_dir): d for d in items_to_process
        }

        for future in tqdm(as_completed(future_to_item), total=len(future_to_item), desc="Processing"):
            try:
                hash_val = future.result()
                # logger.info(f'Processed or skipped: {hash_val}.json')
            except Exception as exc:
                print(f'Generated an exception: {exc}')

